#include <sourcemod>
#include <sdktools>
#include <morecolors>

#pragma semicolon 1

#define PLUGIN_VERSION "2.2.2"

#define LOGO	"{green}[DiacrE Balance]"

#define MAX_FILE_LEN 80

#define KILLS 0
#define DEATHS 1
#define NUM_STATS 2

#define TEAMS_BALANCED 1
#define TEAMS_UNBALANCED 0
#define TEAM_BALANCE_PENDING 3

#define TERRORIST_TEAM 2
#define COUNTER_TERRORIST_TEAM 3
#define NO_TEAM -1
#define NUM_TEAMS 2

#define TERRORIST_INDEX 0
#define COUNTER_TERRORIST_INDEX 1

#define NUM_TEAM_INFOS 3
#define TOTAL_WINS 0
#define CONSECUTIVE_WINS 1
#define ROUND_SWITCH 2

#define ROUNDS_TO_SAVE 6

#define PENDING_MESSAGE 1
#define NOT_BALANCED_MESSAGE 2
#define BALANCED_MESSAGE 4
#define CANT_BALANCE_MESSAGE 8

#define STATS_PER_PANEL 8

#define SQLITE 0
#define MYSQL 1

public Plugin myinfo = 
{
	name = "Team Balance",
	author = "dalto",
	description = "Team Balancer Plugin",
	version = PLUGIN_VERSION,
	url = "http://forums.alliedmods.net"
};

// Global Variables
Handle g_CvarVersion = INVALID_HANDLE;
Handle g_CvarEnabled = INVALID_HANDLE;
Handle g_CvarMinKills = INVALID_HANDLE;
Handle g_CvarConsecutiveWins = INVALID_HANDLE;
Handle g_CvarWinLossRatio = INVALID_HANDLE;
Handle g_CvarRoundsNewJoin = INVALID_HANDLE;
Handle g_CvarMinRounds = INVALID_HANDLE;
Handle g_CvarSaveTime = INVALID_HANDLE;
Handle g_CvarDefaultKDR = INVALID_HANDLE;
Handle g_CvarAnnounce = INVALID_HANDLE;
Handle g_CvarIncrement = INVALID_HANDLE;
Handle g_CvarSingleMax = INVALID_HANDLE;
Handle g_CvarCommands = INVALID_HANDLE;
Handle g_CvarMaintainSize = INVALID_HANDLE;
Handle g_CvarControlJoins = INVALID_HANDLE;
Handle g_CvarDatabase = INVALID_HANDLE;
Handle g_CvarJoinImmunity = INVALID_HANDLE;
Handle g_CvarAdminImmunity = INVALID_HANDLE;
Handle g_CvarMinBalance = INVALID_HANDLE;
Handle g_CvarLockTeams = INVALID_HANDLE;
Handle g_CvarStopSpec = INVALID_HANDLE;
Handle g_CvarAdminFlags = INVALID_HANDLE;
Handle g_CvarLockTime = INVALID_HANDLE;
int roundStats[NUM_TEAMS][ROUNDS_TO_SAVE];
int playerStats[MAXPLAYERS + 1][NUM_STATS];
int whoWonLast;
int teamInfo[NUM_TEAMS][NUM_TEAM_INFOS];
int roundNum;
Handle gameConf = INVALID_HANDLE;
Handle switchTeam = INVALID_HANDLE;
Handle setModel = INVALID_HANDLE;
bool balanceTeams;
bool lateLoaded;
bool forceBalance = false;
static const String:ctModels[4][] = {"models/player/ct_urban.mdl", "models/player/ct_gsg9.mdl", "models/player/ct_sas.mdl", "models/player/ct_gign.mdl"};
static const String:tModels[4][] = {"models/player/t_phoenix.mdl", "models/player/t_leet.mdl", "models/player/t_arctic.mdl", "models/player/t_guerilla.mdl"};
Handle sqlTeamBalanceStats = INVALID_HANDLE;
bool g_commandsHooked = false;
int g_panelPos[MAXPLAYERS + 1];
int g_playerList[MAXPLAYERS];
char g_playerListNames[MAXPLAYERS][40];
int g_playerCount;
int g_dbType;
bool g_switchNextRound[MAXPLAYERS + 1];
int g_balancedLast;
int g_teamList[MAXPLAYERS + 1];
char g_adminFlags[20];
Handle g_kv = INVALID_HANDLE;

public APLRes AskPluginLoad2(Handle myself, bool late, char[] error, int err_max)
{
	lateLoaded = late;
	return APLRes_Success;
}

public OnPluginStart()
{
	LoadTranslations("common.phrases");
	LoadTranslations("diacre_teambalance.phrases");
	
	g_CvarVersion = CreateConVar("sm_team_balance_version", PLUGIN_VERSION, "Team Balance Version", FCVAR_PLUGIN|FCVAR_SPONLY|FCVAR_REPLICATED|FCVAR_NOTIFY);
	g_CvarEnabled = CreateConVar("sm_team_balance_enable", "1", "Enables the Team Balance plugin, when disabled the plugin will still collect stats");
	g_CvarMinKills = CreateConVar("sm_team_balance_min_kd", "10", "The minimum number of kills + deaths in order to be given a real kdr");
	g_CvarAnnounce = CreateConVar("sm_team_balance_announce", "15", "Announcement preferences");
	g_CvarConsecutiveWins = CreateConVar("sm_team_balance_consecutive_wins", "4", "The number of consecutive wins required to declare the teams unbalanced");
	g_CvarWinLossRatio = CreateConVar("sm_team_balance_wlr", "0.55", "The win loss ratio required to declare the teams unbalanced");
	g_CvarRoundsNewJoin = CreateConVar("sm_team_balance_new_join_rounds", "0", "The number of rounds to delay team balancing when a new player joins the losing team");
	g_CvarMinRounds = CreateConVar("sm_team_balance_min_rounds", "2", "The minimum number of rounds before the team balancer starts");
	g_CvarSaveTime = CreateConVar("sm_team_balance_save_time", "672", "The number of hours to save stats for");
	g_CvarDefaultKDR = CreateConVar("sm_team_balance_def_kdr", "1.0", "The default kdr used until a real kdr is established");
	g_CvarIncrement = CreateConVar("sm_team_balance_increment", "5", "The increment for which additional players are balanced");
	g_CvarSingleMax = CreateConVar("sm_team_balance_single_max", "6", "The maximimum number of players on a team for which a single player is balanced");
	g_CvarCommands = CreateConVar("sm_team_balance_commands", "0", "A flag to say whether the team commands will be enabled");
	g_CvarMaintainSize = CreateConVar("sm_team_balance_maintain_size", "1", "A flag to say if the team size should be maintained");
	g_CvarControlJoins = CreateConVar("sm_team_balance_control_joins", "0", "If 1 this plugin fully manages who can join each team");
	g_CvarDatabase = CreateConVar("sm_team_balance_database", "", "The database configuration to use.  Empty for a local SQLite db");
	g_CvarJoinImmunity = CreateConVar("sm_team_balance_join_immunity", "0", "Set to 0 if admins should not be immune to join control");
	g_CvarAdminImmunity = CreateConVar("sm_team_balance_admin_immunity", "0", "0 to disable immunity.  WARNING: Enabling immunity SEVERELY limits the balancing algorithm");
	g_CvarMinBalance = CreateConVar("sm_team_balance_min_balance_frequency", "1", "This is the number of rounds to skip between balancing");
	g_CvarLockTeams = CreateConVar("sm_team_balance_lock_teams", "0", "Set to 1 if you want to force each player to stay in the teams assigned");
	g_CvarStopSpec = CreateConVar("sm_team_balance_stop_spec", "0", "Set to 1 if you don't want players who have already joined a team to be able to switch to spectator");
	g_CvarAdminFlags = CreateConVar("sm_team_balance_admin_flags", "", "The admin flags that admins who should have immunity must have one of");
	g_CvarLockTime = CreateConVar("sm_team_balance_lock_time", "15", "The number of minutes after disconnect before the team lock expires after disconnect");
	
	HookConVarChange(g_CvarAdminFlags, AdminFlagsChanged);
	
	AutoExecConfig(true, "sm_teambalance");
	
	gameConf = LoadGameConfigFile("teambalance.games");
	
	if(gameConf == INVALID_HANDLE)
	{
		SetFailState("gamedata/teambalance.games.txt not loadable");
	}

	StartPrepSDKCall(SDKCall_Player);
	PrepSDKCall_SetFromConf(gameConf, SDKConf_Signature, "SwitchTeam");
	PrepSDKCall_AddParameter(SDKType_PlainOldData, SDKPass_Plain);
	switchTeam = EndPrepSDKCall();

	StartPrepSDKCall(SDKCall_Player);
	PrepSDKCall_SetFromConf(gameConf, SDKConf_Signature, "SetModel");
	PrepSDKCall_AddParameter(SDKType_String, SDKPass_Pointer);
	setModel = EndPrepSDKCall();

	HookEvent("player_death", EventPlayerDeath);
	HookEvent("round_end", EventRoundEnd);
	HookEvent("round_start", EventRoundStart, EventHookMode_Pre);
	HookEvent("player_team", EventPlayerTeamChange);

	InitializeStats();

	RegConsoleCmd("jointeam", CommandJoinTeam);
	RegConsoleCmd("sm_tbstats", CommandKDR);
	RegAdminCmd("sm_tbmenu", TeamManagementMenu, ADMFLAG_CONVARS);
	RegAdminCmd("sm_tbdump", CommandDump, ADMFLAG_GENERIC);
	RegAdminCmd("sm_tbset", CommandSet, ADMFLAG_GENERIC);
	RegAdminCmd("sm_tbswitchatstart", CommandStartSwitch, ADMFLAG_GENERIC);
	RegAdminCmd("sm_tbswitchnow", CommandTeamSwitch, ADMFLAG_GENERIC);
	RegAdminCmd("sm_tbswap", CommandTeamSwap, ADMFLAG_GENERIC);
	
	g_kv=CreateKeyValues("LockExpiration");

	if(lateLoaded)
	{
		for(new i = 1; i < MaxClients; i++)
		{
			if(IsClientInGame(i))
			{
				LoadStats(i);
			}
		}
	}
}

public OnMapStart()
{
	InitializeStats();
	MapStartInitializations();
}

public OnClientPostAdminCheck(client)
{
	if(client && !IsFakeClient(client) && IsClientInGame(client))
	{
		char steamId[30];

		GetClientAuthId(client, AuthId_Steam2, steamId, sizeof(steamId));
		KvRewind(g_kv);
		
		if(KvJumpToKey(g_kv, steamId))
		{
			if(GetTime() < KvGetNum(g_kv, "timestamp") + GetConVarInt(g_CvarLockTime) * 60)
			{
				g_teamList[client] = KvGetNum(g_kv, "team", 0);
				return;
			}
		}
		
		g_teamList[client] = 0;
	}

	LoadStats(client);
}

public OnClientDisconnect(int client)
{
	char steamId[20];
	if(client && !IsFakeClient(client)) 
	{
		GetClientAuthId(client, AuthId_Steam2, steamId, sizeof(steamId));
		KvRewind(g_kv);
		
		if(KvJumpToKey(g_kv, steamId, true))
		{
			KvSetNum(g_kv, "team", g_teamList[client]);
			KvSetNum(g_kv, "timestamp", GetTime());
		}
	}
	
	UpdateStats(client);
}

public EventPlayerDeath(Handle event, const char[] name, bool dontBroadcast)
{
	int victimId = GetEventInt(event, "userid");
	int attackerId = GetEventInt(event, "attacker");
	int attackerClient = GetClientOfUserId(attackerId);
	int victimClient = GetClientOfUserId(victimId);

	if(attackerClient && IsClientInGame(attackerClient))
	{
		playerStats[attackerClient][KILLS]++;
	}
	if(victimClient && IsClientInGame(victimClient))
	{
		playerStats[victimClient][DEATHS]++;
	}
}

public GetTeamBalance()
{
	if(whoWonLast == NO_TEAM || roundNum < GetConVarInt(g_CvarMinRounds))
	{
		return TEAM_BALANCE_PENDING;
	}

	if(roundNum < g_balancedLast + GetConVarInt(g_CvarMinBalance))
	{
		return TEAM_BALANCE_PENDING;
	}

	// check to see if we need to rebalance the teams for size
	if(GetConVarBool(g_CvarMaintainSize))
	{
		// Count the team sizes
		int teamCount[2];
		for(new i = 1; i <= MaxClients; i++)
		{
			if(IsClientInGame(i) && (GetClientTeam(i) == TERRORIST_TEAM || GetClientTeam(i) == COUNTER_TERRORIST_TEAM))
			{
				teamCount[GetTeamIndex(GetClientTeam(i))]++;
			}
		}
		if(teamCount[0] - teamCount[1] > 1 || teamCount[1] - teamCount[0] > 1)
		{
			forceBalance = true;
			return TEAMS_UNBALANCED;
		}
	}
	
	// Check to see if it is pending due to player join
	if(GetConVarInt(g_CvarRoundsNewJoin) && teamInfo[GetOtherTeam(GetTeamIndex(whoWonLast))][ROUND_SWITCH] > roundNum - GetConVarInt(g_CvarRoundsNewJoin))
	{
		return TEAM_BALANCE_PENDING;
	}

	// If the number of consecutive wins has been exceeded than the teams are not balanced
	if(teamInfo[GetTeamIndex(whoWonLast)][CONSECUTIVE_WINS] >= GetConVarInt(g_CvarConsecutiveWins))
	{
		return TEAMS_UNBALANCED;
	}
	
	// Check to see if we are below the minimum winn/loss ratio	
	if(float(teamInfo[GetOtherTeam(GetTeamIndex(whoWonLast))][TOTAL_WINS]) / float(teamInfo[GetTeamIndex(whoWonLast)][TOTAL_WINS]) < GetConVarFloat(g_CvarWinLossRatio))
	{
		return TEAMS_UNBALANCED;
	}

	// check to see if we need to rebalance the teams for size
	if(GetConVarBool(g_CvarMaintainSize))
	{
		// Count the team sizes
		int teamCount[2];
		for(new i = 1; i <= MaxClients; i++)
		{
			if(IsClientInGame(i) && (GetClientTeam(i) == TERRORIST_TEAM || GetClientTeam(i) == COUNTER_TERRORIST_TEAM))
			{
				teamCount[GetTeamIndex(GetClientTeam(i))]++;
			}
		}
		if(teamCount[0] - teamCount[1] > 1 || teamCount[1] - teamCount[0] > 1)
		{
			forceBalance = true;
			return TEAMS_UNBALANCED;
		}
	}
	
	// If we are not unbalanced or pending then we must be balanced
	return TEAMS_BALANCED;
}

public EventRoundEnd(Handle event, const char[] name, bool dontBroadcast)
{
	int winner = GetEventInt(event, "winner");
	int reason = GetEventInt(event, "reason");
	
	if(reason == 16)
	{
		return Plugin_Handled;
	}
	
	if(winner != TERRORIST_TEAM && winner != COUNTER_TERRORIST_TEAM)
	{
		return Plugin_Handled;
	}
	
	// Update the map statistics
	roundStats[GetTeamIndex(winner)][roundNum % ROUNDS_TO_SAVE] = 1;
	roundStats[GetTeamIndex(GetOtherTeam(winner))][roundNum % ROUNDS_TO_SAVE] = 0;
	// We need recent statistics for our win-loss-ratio comparison
	teamInfo[COUNTER_TERRORIST_INDEX][TOTAL_WINS] = 0;
	teamInfo[TERRORIST_INDEX][TOTAL_WINS] = 0;
	for(new i = 0; i < ROUNDS_TO_SAVE; i++)
	{
		teamInfo[COUNTER_TERRORIST_INDEX][TOTAL_WINS] += roundStats[COUNTER_TERRORIST_INDEX][i];
		teamInfo[TERRORIST_INDEX][TOTAL_WINS] += roundStats[TERRORIST_INDEX][i];
	}	

	// update whoWonLast and consecutive wins counts
	teamInfo[GetTeamIndex(winner)][CONSECUTIVE_WINS]++;
	if(whoWonLast != winner)
	{
		whoWonLast = winner;
		teamInfo[GetTeamIndex(GetOtherTeam(winner))][CONSECUTIVE_WINS] = 0;
	}
	
	if(GetConVarBool(g_CvarEnabled))
	{
		// Check to see if the teams are in balance and take action as needed
		switch(GetTeamBalance())
		{
			case TEAM_BALANCE_PENDING:
			{
				if(GetConVarInt(g_CvarAnnounce) & PENDING_MESSAGE)
				{
					PrintTranslatedToChatAll("pending");
				}
			}
			case TEAMS_BALANCED:
			{
				if(GetConVarInt(g_CvarAnnounce) & BALANCED_MESSAGE)
				{
					PrintTranslatedToChatAll("balanced");
				}
			}
			case TEAMS_UNBALANCED:
			{
				if(GetConVarInt(g_CvarAnnounce) & NOT_BALANCED_MESSAGE)
				{
					PrintTranslatedToChatAll("not balanced");
				}
				balanceTeams = true;
			}
		}
		
		for (int i = 1; i <= MaxClients; i++)
		{
			if (g_switchNextRound[i] && GetClientTeam(i) > 1)
			{
				SwitchTeam(i, GetOtherTeam(GetClientTeam(i)));
				g_switchNextRound[i] = false;
			}
		}
		
		if (balanceTeams)
		{
			if(!BalanceTeams() && GetConVarInt(g_CvarAnnounce) & CANT_BALANCE_MESSAGE)
			{
				PrintTranslatedToChatAll("unbalanceable");
			}
			balanceTeams = false;
			forceBalance = false;
		}
	}

	roundNum++;
	
	CreateTimer(1.5, protectAllDueToSwitch);
	
	return Plugin_Continue;
}

public Action protectAllDueToSwitch(Handle timer)
{
	for (int i = 1; i <= MaxClients; i++)
	{
		if (IsValidAndAlive(i))
		{
			protect(i);
		}
	}
	
	return Plugin_Continue;
}

public EventPlayerTeamChange(Handle event, const char[] name, bool dontBroadcast)
{
	int userid = GetEventInt(event, "userid");
	int team = GetEventInt(event, "team");
	int client = GetClientOfUserId(userid);
	
	if(client && team == COUNTER_TERRORIST_TEAM || team == TERRORIST_TEAM && IsClientInGame(client) && !IsFakeClient(client))
	{
		teamInfo[GetTeamIndex(team)][ROUND_SWITCH] = roundNum;
		g_teamList[client] = team;
	}
	
	return Plugin_Continue;
}

public Action EventRoundStart(Handle event, const char[] name, bool dontBroadcast)
{
	// Update the player list
	g_playerCount = 0;
	for (int i = 1; i <= MaxClients; i++)
	{
		if(IsClientInGame(i))
		{
			unprotect(i);
			g_playerList[g_playerCount] = i;
			g_playerCount++;
		}
	}
	
	// Sort the player list by KDR
	SortCustom1D(g_playerList, g_playerCount, SortKDR);
	
	for (int i = 0; i < g_playerCount; i++)
	{
		GetClientName(g_playerList[i], g_playerListNames[i], sizeof(g_playerListNames[]));
	}
	
	return Plugin_Continue;
}


//  Balances the teams based on KDR
public bool BalanceTeams()
{
	int arrayTeams[2][MAXPLAYERS];
	int switchArray[MAXPLAYERS];
	int bottomPlayer;
	int numPlayers;
	int clientToSwitch;
	char buffer[40];

	// Put all the players into arrays by team
	int teamCount[2];
	for(new i = 1; i <= MaxClients; i++)
	{
		if(IsClientInGame(i) && (GetClientTeam(i) == TERRORIST_TEAM || GetClientTeam(i) == COUNTER_TERRORIST_TEAM))
		{
			arrayTeams[GetTeamIndex(GetClientTeam(i))][teamCount[GetTeamIndex(GetClientTeam(i))]] = i;
			teamCount[GetTeamIndex(GetClientTeam(i))]++;
		}
	}
	
	// Sort the arrays by KDR
	SortCustom1D(arrayTeams[TERRORIST_INDEX], teamCount[TERRORIST_INDEX], SortKDR);
	SortCustom1D(arrayTeams[COUNTER_TERRORIST_INDEX], teamCount[COUNTER_TERRORIST_INDEX], SortKDR);
	
	// If there is only one person on the winning team there is not much we can do to fix the situation
	if(teamCount[GetTeamIndex(whoWonLast)] <= 1  && !forceBalance)
	{
		return false;
	}
		
	// Decide how many people to switch
	if(teamCount[GetTeamIndex(whoWonLast)] - GetConVarInt(g_CvarSingleMax) <= 0)
	{
		numPlayers = 1;
	} else {
		numPlayers = ((teamCount[GetTeamIndex(whoWonLast)] - GetConVarInt(g_CvarSingleMax)) / GetConVarInt(g_CvarIncrement)) + 1;
	}
	
	if(teamCount[GetTeamIndex(whoWonLast)] - teamCount[GetTeamIndex(GetOtherTeam(whoWonLast))] > numPlayers)
	{
		numPlayers = teamCount[GetTeamIndex(whoWonLast)] - teamCount[GetTeamIndex(GetOtherTeam(whoWonLast))];
	}

	// The first player available for switching.  1 is the second best player on the team
	clientToSwitch = 1;

	// Check to make sure the switches we are doing are going to be positive changes
	int goodPlayers = 0;
	bottomPlayer = teamCount[GetTeamIndex(GetOtherTeam(whoWonLast))] - 1;
	float lowKDR = GetKDR(arrayTeams[GetTeamIndex(GetOtherTeam(whoWonLast))][bottomPlayer]);
	for (int i = 0; i < numPlayers; i++)
	{
		if(GetKDR(arrayTeams[GetTeamIndex(whoWonLast)][i + clientToSwitch]) > lowKDR)
		{
			goodPlayers++;
			if(bottomPlayer > 0)
			{
				bottomPlayer--;
				lowKDR = GetKDR(arrayTeams[GetTeamIndex(GetOtherTeam(whoWonLast))][bottomPlayer]);
			}
		}
	}
	
	// goodPlayers now contains a revised number of players to switch
	// if it is 0 than the teams are as balanced as possible without stacking
	numPlayers = goodPlayers;
	
	// check to make sure the winning team isn't significantly larger
	int minPlayers = RoundToCeil(float(teamCount[GetTeamIndex(whoWonLast)] - teamCount[GetTeamIndex(GetOtherTeam(whoWonLast))]) / 2.0);
	if(numPlayers < minPlayers)
	{
		numPlayers = minPlayers;
	}

	if(numPlayers == 0 && !forceBalance)
	{
		return false;
	}
	
	// Set the switchArray to switched for all indexes after the last player
	for(new i = teamCount[GetTeamIndex(whoWonLast)]; i < MAXPLAYERS; i++)
	{
		switchArray[i] = 1;
	}
	
	// If admin immunity is one make admins immune
	for(new i = 0; i < teamCount[GetTeamIndex(whoWonLast)]; i++)
	{
		if(GetConVarBool(g_CvarAdminImmunity) && IsAdmin(arrayTeams[GetTeamIndex(whoWonLast)][i]))
		{
			switchArray[i] = 1;
		}
	}
	
	// Do the team switching
	bool found = true;
	int switched = 0;
	for (int i = 0; i < numPlayers; i++)
	{
		// If we have already switched this player we need to switch someone else
		// so we keep getting the next lowest player until we find someone we have not switched
		while (switchArray[clientToSwitch] && found)
		{
			if(clientToSwitch < teamCount[GetTeamIndex(whoWonLast)] - 1)
			{
				clientToSwitch++;
			} else {
				clientToSwitch = 1;
			}
			if(GetConVarBool(g_CvarAdminImmunity))
			{
				found = false;
				for(new player = 1; player < MAXPLAYERS; player++)
				{
					if(switchArray[player] == 0)
					{
						found = true;
					}
				}
			}	
		}
			
		// Switch the team of the player on the winning team
		if (found)
		{
			SwitchTeam(arrayTeams[GetTeamIndex(whoWonLast)][clientToSwitch], GetOtherTeam(whoWonLast));
			switchArray[clientToSwitch] = 1;
			GetClientName(arrayTeams[GetTeamIndex(whoWonLast)][clientToSwitch], buffer, 30);
			switched++;
		
			// Decide who to switch next
			if(i % 2)
			{
				clientToSwitch -= GetConVarInt(g_CvarIncrement) - 1;
			} else {
				clientToSwitch += GetConVarInt(g_CvarIncrement);
			}
			
			if(clientToSwitch >= MaxClients  || clientToSwitch <= 0)
			{
				clientToSwitch = 1;
			}
		}
	}
	
	// Find the worst player on the losing team
	bottomPlayer = teamCount[GetTeamIndex(GetOtherTeam(whoWonLast))] - 1;
	
	// Adjust the team count for how many people were switched
	teamCount[GetTeamIndex(whoWonLast)] -= switched;
	teamCount[GetTeamIndex(GetOtherTeam(whoWonLast))] += switched;
	
	// We remove the worse players from losing team to make the losing team no more than 
	// one client larger than the winning team.
	while(teamCount[GetTeamIndex(whoWonLast)] + 1 < teamCount[GetTeamIndex(GetOtherTeam(whoWonLast))] && bottomPlayer >= 0)
	{
		if(!(GetConVarBool(g_CvarAdminImmunity) && IsAdmin(arrayTeams[GetTeamIndex(GetOtherTeam(whoWonLast))][bottomPlayer])))
		{
			SwitchTeam(arrayTeams[GetTeamIndex(GetOtherTeam(whoWonLast))][bottomPlayer], whoWonLast);
			teamCount[GetTeamIndex(GetOtherTeam(whoWonLast))]--;
			teamCount[GetTeamIndex(whoWonLast)]++;
		}
		bottomPlayer--;
	}
	
	g_balancedLast = roundNum;
	return true;
}

// Finds the other team
// Accepts an index or a team
public GetOtherTeam(int team)
{
	switch(team)
	{
		case 0:
			return 1;
		case 1:
			return 0;
		case 2:
			return 3;
		case 3:
			return 2;
	}
	
	return -1;
}

// Given a team id returns the matching index
public GetTeamIndex(int team)
{
	return team - 2;
}

// The comparison function used in the sort routine in balance teams
// Use to sort an array of clients by KDR
public SortKDR(elem1, elem2, const array[], Handle hndl)
{
	if(GetKDR(elem1) > GetKDR(elem2))
	{
		return -1;
	} else if(GetKDR(elem1) == GetKDR(elem2)) {
		return 0;
	} else {
		return 1;
	}
}

// Switches the team of a client and associated a random player model
// Also sends a message to let the player know they have been switched
// Based on the team switch command from SM Super Commands by pRED*
public SwitchTeam(int client, int team)
{
	// Switch the players team
	SDKCall(switchTeam, client, team);
	
	// Set a random model
	int random = GetRandomInt(0, 3);
	
	if(team == TERRORIST_TEAM)
	{
		SDKCall(setModel, client, tModels[random]);
		PrintCenterText(client, "%t", "t switch");
	} 
	else if(team == COUNTER_TERRORIST_TEAM) 
	{
		SDKCall(setModel, client, ctModels[random]);
		PrintCenterText(client, "%t", "ct switch");
	}
	
	if (IsValid(client))
	{
		CPrintToChatAll("%s: %t", LOGO, "DIACRE_ClientChangeTeam", client);
	}
}

// Swaps the entirety of one team to another
public Action CommandTeamSwap(int client, int args)
{
	for(int i = 1; i <= MaxClients; i++)
	{
		if(IsClientInGame(i) && (GetClientTeam(i) == TERRORIST_TEAM || GetClientTeam(i) == COUNTER_TERRORIST_TEAM))
		{
			SwitchTeam(i, GetOtherTeam(GetClientTeam(i)));
		}
	}
	
	return Plugin_Handled;
}

// Here we get a handle to the database and create it if it doesn't already exist
public InitializeStats()
{
	char error[255];
	char connection[50];
	
	// g_CvarDatabase stores the default connection profile
	GetConVarString(g_CvarDatabase, connection, sizeof(connection));
	
	if(StrEqual(connection, ""))
	{
		// if connection is "" then we use the legacy sqlite db
		sqlTeamBalanceStats = SQL_ConnectEx(SQL_GetDriver("sqlite"), "", "", "", "team_balance", error, sizeof(error), true, 0);
	} 
	else 
	{
		// otherwise we use the record from the configuration
		sqlTeamBalanceStats = SQL_Connect(connection, true, error, sizeof(error)); 
	}
	if(sqlTeamBalanceStats == INVALID_HANDLE)
	{
		SetFailState(error);
	}
	
	char driver[30];
	SQL_ReadDriver(sqlTeamBalanceStats, driver, sizeof(driver));
	
	if(StrEqual(driver, "sqlite"))
	{
		g_dbType = SQLITE;
	} 
	else if(StrEqual(driver, "mysql")) 
	{
		g_dbType = MYSQL;
	} 
	else 
	{
		SetFailState("Only MySQL and SQLite are currently supported");
	}
		
	SQL_LockDatabase(sqlTeamBalanceStats);
	
	if(g_dbType == SQLITE)
	{
		SQL_FastQuery(sqlTeamBalanceStats, "CREATE TABLE IF NOT EXISTS stats (steam_id TEXT, kills INTEGER, deaths INTEGER, kdr REAL, timestamp INTEGER);");
		SQL_FastQuery(sqlTeamBalanceStats, "CREATE UNIQUE INDEX IF NOT EXISTS stats_steam_id on stats (steam_id);");
	} 
	else 
	{
		SQL_FastQuery(sqlTeamBalanceStats, "CREATE TABLE IF NOT EXISTS stats (steam_id VARCHAR(50), kills INTEGER, deaths INTEGER, kdr FLOAT, timestamp INTEGER, PRIMARY KEY(steam_id));");
		
		if(SQL_GetError(sqlTeamBalanceStats, error, sizeof(error)))
		{
			SetFailState(error);
		}
	}
	
	SQL_UnlockDatabase(sqlTeamBalanceStats);
}

// Load the stats for a given client
public LoadStats(int client)
{
	if(!client)
	{
		return;
	}
		
	char steamId[20];
	GetSteamId(client, steamId, sizeof(steamId));

	char buffer[200];
	Format(buffer, sizeof(buffer), "SELECT kills, deaths, kdr, timestamp FROM stats WHERE steam_id = '%s'", steamId);
	SQL_TQuery(sqlTeamBalanceStats, LoadStatsCallback, buffer, client);
}

public LoadStatsCallback(Handle owner, Handle hndl, const char[] error, any data)
{
	if(!StrEqual("", error))
	{
		LogError("Update Stats SQL Error: %s", error);
		return;
	}
	
	int client = data;
	
	if(SQL_FetchRow(hndl))
	{
		if(SQL_FetchInt(hndl, 3) > GetTime() - GetConVarInt(g_CvarSaveTime) * 3600)
		{
			playerStats[client][KILLS] = SQL_FetchInt(hndl, 0);
			playerStats[client][DEATHS] = SQL_FetchInt(hndl, 1);
			return;
		}
	}
	
	playerStats[client][DEATHS] = 0;
	playerStats[client][KILLS] = 0;
}

// Updates the database for a single client
public UpdateStats(int client)
{
	char steamId[20];
	char buffer[255];
	
	if(IsClientInGame(client))
	{
		GetSteamId(client, steamId, sizeof(steamId));

		Format(buffer, sizeof(buffer), "REPLACE INTO stats VALUES ('%s', %i, %i, %f, %i)", steamId, playerStats[client][KILLS], playerStats[client][DEATHS], GetKDR(client), GetTime());
		SQL_TQuery(sqlTeamBalanceStats, SQLErrorCheckCallback, buffer);
	}
}

// This is used during a threaded query that does not return data
public SQLErrorCheckCallback(Handle owner, Handle hndl, const char[] error, any data)
{
	if(!StrEqual("", error))
	{
		LogError("Team Balance SQl Error: %s", error);
	}
}


// We actually want to track bot stats for our purposes
// so we give them fake steam id's
public GetSteamId(int client, char[] buffer, int bufferSize)
{
	if(!client)
	{
		return;
	}
		
	if(IsFakeClient(client))
	{
		GetClientName(client, buffer, bufferSize);
		return;
	}

	GetClientAuthId(client, AuthId_Steam2, buffer, bufferSize);
}

public PrintTranslatedToChatAll(char[] buffer)
{
	for (int i = 1; i <= MaxClients; i++)
	{
		if(IsClientInGame(i) && !IsFakeClient(i))
		{
			CPrintToChat(i, "%s:%t", LOGO, buffer);
		}
	}
}

public MapStartInitializations()
{
	whoWonLast = NO_TEAM;
	teamInfo[TERRORIST_INDEX][CONSECUTIVE_WINS] = 0;
	teamInfo[TERRORIST_INDEX][ROUND_SWITCH] = 0;
	teamInfo[COUNTER_TERRORIST_INDEX][CONSECUTIVE_WINS] = 0;
	teamInfo[COUNTER_TERRORIST_INDEX][ROUND_SWITCH] = 0;
	
	for(new i = 0; i < ROUNDS_TO_SAVE; i++)
	{
		roundStats[COUNTER_TERRORIST_INDEX][i] = 0;
		roundStats[TERRORIST_INDEX][i] = 0;
	}
	
	roundNum = 1;
	balanceTeams = false;
	g_balancedLast = 0;
	Prune();
}

public OnConfigsExecuted()
{
	GetConVarString(g_CvarAdminFlags, g_adminFlags, sizeof(g_adminFlags));
	
	if(GetConVarBool(g_CvarCommands) && !g_commandsHooked)
	{
		RegAdminCmd("sm_swapteams", CommandTeamSwap, ADMFLAG_GENERIC);
		RegAdminCmd("sm_teamswitch", CommandTeamSwitch, ADMFLAG_GENERIC);
		
		g_commandsHooked = true;
	}
}

public Action CommandTeamSwitch(int client, int args)
{
	if(args != 1)
	{
		ReplyToCommand(client, "Usage: sm_teamswitch <player>");
		return Plugin_Handled;
	}

	int target;
	char buffer[50];
	GetCmdArg(1, buffer, sizeof(buffer));	
	target = FindTarget(client, buffer, false, false);
	
	if(target && target != -1 && (GetClientTeam(target) == TERRORIST_TEAM || GetClientTeam(target) == COUNTER_TERRORIST_TEAM))
	{
		SwitchTeam(target, GetOtherTeam(GetClientTeam(target)));
	}

	return Plugin_Handled;
}

public Action CommandJoinTeam(int client, int args)
{
	if(!GetConVarBool(g_CvarControlJoins))
	{
		return Plugin_Continue;
	}

	if(args != 1)
	{
		return Plugin_Continue;
	}

	if(GetConVarBool(g_CvarJoinImmunity) && IsAdmin(client))
	{
		return Plugin_Continue;
	}

	char teamString[3];
	GetCmdArg(1, teamString, sizeof(teamString));
	int team = StringToInt(teamString);
	
	int curTeam = GetClientTeam(client);
	
	if(curTeam > 1 && team == 1 && GetConVarBool(g_CvarStopSpec))
	{
		PrintCenterText(client, "Only players who have not joined a team may spectate");
		return Plugin_Handled;
	}
	
	// Check for team locking concerns
	if(curTeam != 0 && GetConVarBool(g_CvarLockTeams) && team != 1 && g_teamList[client])
	{
		// if autojoin force them back onto their team
		if(team == 0)
		{
			ChangeClientTeam(client, g_teamList[client]);
			return Plugin_Handled;
		}
		
		// check to see if the team they are switching to is the same as their assigned team
		if(team != g_teamList[client])
		{
			PrintCenterText(client, "You cannot join that team");
			return Plugin_Handled;
		}
		
		// if we get to here it is safe
		return Plugin_Continue;
	}

	// We only want to get in the way of people joining these teams
	if(team != TERRORIST_TEAM && team != COUNTER_TERRORIST_TEAM)
	{
		return Plugin_Continue;
	}
	
	// Count the team sizes
	int teamCount[2];
	for(new i = 1; i <= MaxClients; i++)
	{
		if(IsClientInGame(i) && (GetClientTeam(i) == TERRORIST_TEAM || GetClientTeam(i) == COUNTER_TERRORIST_TEAM))
		{
			teamCount[GetTeamIndex(GetClientTeam(i))]++;
		}
	}
	if(teamCount[GetTeamIndex(TERRORIST_TEAM)] > teamCount[GetTeamIndex(COUNTER_TERRORIST_TEAM)])
	{
		// The terrorist team is bigger so joining the CT's is fine
		if(team == COUNTER_TERRORIST_TEAM)
		{
			return Plugin_Continue;
		}
		
		if(curTeam == COUNTER_TERRORIST_TEAM)
		{
			PrintCenterText(client, "You are not allowed to join the stronger team");
			return Plugin_Handled;
		}
		PrintCenterText(client, "The admin is joining you to the Counter-Terrorist team");
		ChangeClientTeam(client, 3);
		return Plugin_Handled;
	} else if(teamCount[GetTeamIndex(TERRORIST_TEAM)] < teamCount[GetTeamIndex(COUNTER_TERRORIST_TEAM)]) {
		// The counter terrorist team is bigger so joining the T's is fine
		if(team == TERRORIST_TEAM)
		{
			return Plugin_Continue;
		}
		
		if(curTeam == TERRORIST_TEAM)
		{
			PrintCenterText(client, "You are not allowed to join the stronger team");
			return Plugin_Handled;
		}
		PrintCenterText(client, "The admin is joining you to the Terrorist team");
		ChangeClientTeam(client, 2);
		return Plugin_Handled;
	} else {
		// The teams are equal
		
		// if they are already on an existing team than they can't switch
		if(curTeam == TERRORIST_TEAM || curTeam == COUNTER_TERRORIST_TEAM)
		{
			PrintCenterText(client, "You are not allowed to switch to the other team right now");
			return Plugin_Handled;
		}

		// The teams are equal and we don't enough stats to know what it up so let them do what they want
		if(roundNum <= 2)
		{
			return Plugin_Continue;
		}
					
		// Count up the times the CT's have one in the last x rounds
		int ctWins;
		for(new i = 1; i <= 2; i++) {
			ctWins += roundStats[GetTeamIndex(COUNTER_TERRORIST_TEAM)][(roundNum - i) % ROUNDS_TO_SAVE];
		}
		
		// if the CT's are winning and the player is trying to join the CT's...they can't
		if(ctWins == 2 && team == 3)
		{
			PrintCenterText(client, "The admin is joining you to the Terrorist team");
			ChangeClientTeam(client, 2);
			return Plugin_Handled;
		}
			
		// If the T's are winning and the player is trying to join the CT team....he's screwed
		if(ctWins == 0 && team == 2)
		{
			PrintCenterText(client, "The admin is joining you to the Counter-Terrorist team");
			ChangeClientTeam(client, 3);
			return Plugin_Handled;
		}
		return Plugin_Continue;
	}
}

public Action TeamManagementMenu(int client, int args)
{
	Handle menu = CreateMenu(TeamManagementMenuHandler);
	char buffer[100];
	Format(buffer, sizeof(buffer), "%T", "team management menu", client);
	SetMenuTitle(menu, buffer);
	
	Format(buffer, sizeof(buffer), "%T", "swap teams", client);
	AddMenuItem(menu, "team menu item", buffer);
	
	Format(buffer, sizeof(buffer), "%T", "switch player", client);
	AddMenuItem(menu, "team menu item", buffer);
	
	Format(buffer, sizeof(buffer), "%T", "switch player next", client);
	AddMenuItem(menu, "team menu item", buffer);
	
	if(GetConVarBool(g_CvarMaintainSize))
	{
		Format(buffer, sizeof(buffer), "%T", "maintain size off", client);
	} else {
		Format(buffer, sizeof(buffer), "%T", "maintain size on", client);
	}
	AddMenuItem(menu, "team menu item", buffer);
	
	if(GetConVarBool(g_CvarControlJoins))
	{
		Format(buffer, sizeof(buffer), "%T", "control joins off", client);
	} else {
		Format(buffer, sizeof(buffer), "%T", "control joins on", client);
	}
	AddMenuItem(menu, "team menu item", buffer);
	
	if(GetConVarBool(g_CvarEnabled))
	{
		Format(buffer, sizeof(buffer), "%T", "disable balancer", client);
	} else {
		Format(buffer, sizeof(buffer), "%T", "enable balancer", client);
	}
	AddMenuItem(menu, "team menu item", buffer);

	Format(buffer, sizeof(buffer), "%T", "display stats", client);
	AddMenuItem(menu, "team menu item", buffer);

	Format(buffer, sizeof(buffer), "%T", "dump settings", client);
	AddMenuItem(menu, "team menu item", buffer);
	
	SetMenuExitButton(menu, true);

	DisplayMenu(menu, client, 20);
 
	return Plugin_Handled;
}

//  This handles the selling
public TeamManagementMenuHandler(Handle menu, MenuAction action, int param1, int param2)
{
	if(action == MenuAction_Select)	{
		switch(param2)
		{
			// swap teams
			case 0:
			{
				CommandTeamSwap(param1, 0);
			}
			// switch player
			case 1:
			{
				SwitchPlayerMenu(param1);
				return;
			}
			// switch next round
			case 2:
			{
				SwitchPlayerMenu(param1, true);
				return;
			}
			// maintain size
			case 3:
			{
				SetConVarBool(g_CvarMaintainSize, FlipBool(GetConVarBool(g_CvarMaintainSize)));
			}
			// control joins
			case 4:
			{
				SetConVarBool(g_CvarControlJoins, FlipBool(GetConVarBool(g_CvarControlJoins)));
			}
			// disable/enable
			case 5:
			{
				SetConVarBool(g_CvarEnabled, FlipBool(GetConVarBool(g_CvarEnabled)));
			}
			// display stats
			case 6:
			{
				CommandKDR(param1, 0);
				return;
			}
			// dump settings
			case 7:
			{
				CommandDump(param1, 0);
			}
		}
		TeamManagementMenu(param1, 0);
	} else if(action == MenuAction_End)	{
		CloseHandle(menu);
	}
}

public bool FlipBool(bool current)
{
	if(current)
		return false;
	else
		return true;
}

SwitchPlayerMenu(client, bool nextRound = false)
{
	Handle menu = CreateMenu(SwitchPlayerMenuHandler);
	char buffer[100];
	char clientString[4];
	if(nextRound)
	{
		Format(buffer, sizeof(buffer), "%T", "switch player next", client);
	} 
	else 
	{
		Format(buffer, sizeof(buffer), "%T", "switch player", client);
	}
	
	SetMenuTitle(menu, buffer);
	
	for(new i = 1; i <= MaxClients; i++)
	{
		if(IsClientInGame(i) && GetClientTeam(i) > 1)
		{
			IntToString(i, clientString, sizeof(clientString));
			GetClientName(i, buffer, sizeof(buffer));
			if(nextRound)
			{
				Format(clientString, sizeof(clientString), "N%s", clientString);
			}
			AddMenuItem(menu, clientString, buffer);
		}
	}
	
	SetMenuExitButton(menu, true);

	DisplayMenu(menu, client, 20);
 
	return;
}

public SwitchPlayerMenuHandler(Handle menu, MenuAction action, int param1, int param2)
{
	if(action == MenuAction_Select)
	{
		bool nextRound = false;
		char playerString[4];
		decl player;
		GetMenuItem(menu, param2, playerString, sizeof(playerString));
		
		if(playerString[0] == 'N')
		{
			nextRound = true;
			player = StringToInt(playerString[1]);
		} 
		else 
		{
			player = StringToInt(playerString);
		}
		if(player && IsClientInGame(player) && GetClientTeam(player) > 1)
		{
			if(nextRound)
			{
				g_switchNextRound[player] = true;
			} 
			else 
			{
				SwitchTeam(player, GetOtherTeam(GetClientTeam(player)));
			}
		}
		SwitchPlayerMenu(param1, nextRound);
	} 
	else if(action == MenuAction_End)	
	{
		CloseHandle(menu);
	}
}

public Action CommandKDR(int client, int args)
{
	if(client && IsClientInGame(client))
	{
		KDRPanel(client, 1);
	}
	
	return Plugin_Handled;
}

public float GetKDR(int client)
{
	if(!client || !IsClientInGame(client))
	{
		return 0.0;
	}
	
	if(playerStats[client][KILLS] + playerStats[client][DEATHS] >= GetConVarInt(g_CvarMinKills)) 
	{
		if(playerStats[client][DEATHS])
		{
			return float(playerStats[client][KILLS]) / float(playerStats[client][DEATHS]);
		} 
		else 
		{
			return float(playerStats[client][KILLS]);
		}
	}
	
	return GetConVarFloat(g_CvarDefaultKDR);
}

public KDRPanel(client, panelNumber)
{
	Handle panel = CreatePanel();
	char buffer[512];
	Format(buffer, sizeof(buffer), "%T", "kdr panel", client);
	SetPanelTitle(panel, buffer);
	
	Format(buffer, sizeof(buffer), "%T\n", "internal stats", client);
	for(new i = 0 + (panelNumber - 1) * STATS_PER_PANEL; i < STATS_PER_PANEL * panelNumber; i++)
	{
		if(i < g_playerCount)
		{
			Format(	buffer,
					sizeof(buffer),
					"%s%i: %s - Kills: %i Deaths: %i KDR: %.2f\n",
					buffer,
					i + 1,
					g_playerListNames[i],
					playerStats[g_playerList[i]][KILLS],
					playerStats[g_playerList[i]][DEATHS],
					GetKDR(g_playerList[i]));
		}
	}
	
	DrawPanelItem(panel, buffer);
	if(panelNumber > 1)
	{
 		SetPanelCurrentKey(panel, 8);
 		DrawPanelItem(panel, "Previous");
	}		

 	if(panelNumber * STATS_PER_PANEL < GetClientCount())
 	{
 		SetPanelCurrentKey(panel, 9);
 		DrawPanelItem(panel, "Next");
 	}
	g_panelPos[client] = panelNumber;
 	
 	SetPanelCurrentKey(panel, 10);
 	DrawPanelItem(panel, "Exit");
	SendPanelToClient(panel, client, KDRPanelHandler, 20);

 
	CloseHandle(panel);
}

public KDRPanelHandler(Handle menu, MenuAction action, int param1, int param2)
{
	if(action == MenuAction_Select)
	{
		switch(param2)
		{
			case 8:
				KDRPanel(param1, g_panelPos[param1] - 1);
			case 9:
				KDRPanel(param1, g_panelPos[param1] + 1);
		}
	}
}

public Action CommandDump(int client, int args)
{
	char buffer[250];
	
	GetConVarString(g_CvarVersion, buffer, sizeof(buffer));
	ReplyToCommand(client, "sm_team_balance_version: %s", buffer);
	ReplyToCommand(client, "sm_team_balance_enable: %i", GetConVarInt(g_CvarEnabled));
	ReplyToCommand(client, "sm_team_balance_min_kd: %i", GetConVarInt(g_CvarMinKills));
	ReplyToCommand(client, "sm_team_balance_announce: %i", GetConVarInt(g_CvarAnnounce));
	ReplyToCommand(client, "sm_team_balance_consecutive_wins: %i", GetConVarInt(g_CvarConsecutiveWins));
	ReplyToCommand(client, "sm_team_balance_wlr: %f", GetConVarFloat(g_CvarWinLossRatio));
	ReplyToCommand(client, "sm_team_balance_new_join_rounds: %i", GetConVarInt(g_CvarRoundsNewJoin));
	ReplyToCommand(client, "sm_team_balance_min_rounds: %i", GetConVarInt(g_CvarMinRounds));
	ReplyToCommand(client, "sm_team_balance_save_time: %i", GetConVarInt(g_CvarSaveTime));
	ReplyToCommand(client, "sm_team_balance_def_kdr: %f", GetConVarFloat(g_CvarDefaultKDR));
	ReplyToCommand(client, "sm_team_balance_increment: %i", GetConVarInt(g_CvarIncrement));
	ReplyToCommand(client, "sm_team_balance_single_max: %i", GetConVarInt(g_CvarSingleMax));
	ReplyToCommand(client, "sm_team_balance_commands: %i", GetConVarInt(g_CvarCommands));
	ReplyToCommand(client, "sm_team_balance_maintain_size: %i", GetConVarInt(g_CvarMaintainSize));
	ReplyToCommand(client, "sm_team_balance_control_joins: %i", GetConVarInt(g_CvarControlJoins));
	GetConVarString(g_CvarDatabase, buffer, sizeof(buffer));
	ReplyToCommand(client, "sm_team_balance_database: %s", buffer);
	ReplyToCommand(client, "sm_team_balance_join_immunity: %i", GetConVarInt(g_CvarJoinImmunity));
	ReplyToCommand(client, "sm_team_balance_admin_immunity: %i", GetConVarInt(g_CvarAdminImmunity));
	GetConVarString(g_CvarAdminFlags, buffer, sizeof(buffer));
	ReplyToCommand(client, "sm_team_balance_admin_flags: %s", buffer);
	ReplyToCommand(client, "sm_team_balance_min_balance_frequency: %i", GetConVarInt(g_CvarMinBalance));
	ReplyToCommand(client, "sm_team_balance_lock_teams: %i", GetConVarInt(g_CvarLockTeams));
	ReplyToCommand(client, "sm_team_balance_lock_time: %i", GetConVarInt(g_CvarLockTime));
	ReplyToCommand(client, "sm_team_balance_stop_spec: %i", GetConVarInt(g_CvarStopSpec));

	return Plugin_Handled;
}

public Action CommandSet(int client, int args)
{
	if(args != 3)
	{
		ReplyToCommand(client, "Usage: sm_tbset <player> <kills> <deaths>");
		return Plugin_Handled;
	}

	int target, kills, deaths;
	char buffer[50];
	GetCmdArg(1, buffer, sizeof(buffer));	
	target = FindTarget(client, buffer, false, false);
	GetCmdArg(2, buffer, sizeof(buffer));
	kills = StringToInt(buffer);
	GetCmdArg(3, buffer, sizeof(buffer));
	deaths = StringToInt(buffer);
	
	if(target && target != -1)
	{
		playerStats[target][KILLS] = kills;
		playerStats[target][DEATHS] = deaths;
		GetClientName(target, buffer, sizeof(buffer));
		ReplyToCommand(client, "Set %s's stats to kills: %i, deaths: %i", buffer, playerStats[target][KILLS], playerStats[target][DEATHS]);
	} 
	else 
	{
		ReplyToCommand(client, "Target not found");
	}

	return Plugin_Handled;
}

public Action CommandStartSwitch(int client, int args)
{
	if(args != 1)
	{
		ReplyToCommand(client, "Usage: sm_tbswitchatstart <player>");
		return Plugin_Handled;
	}

	int target;
	char buffer[50];
	GetCmdArg(1, buffer, sizeof(buffer));	
	target = FindTarget(client, buffer, false, false);
	
	if(target && target != -1 && (GetClientTeam(target) == TERRORIST_TEAM || GetClientTeam(target) == COUNTER_TERRORIST_TEAM))
	{
		g_switchNextRound[target] = true;
	}

	return Plugin_Handled;
}

bool IsAdmin(int client)
{
	if(StrEqual(g_adminFlags, ""))
	{
		return GetUserAdmin(client) == INVALID_ADMIN_ID ? false : true;
	}
	
	return (GetUserFlagBits(client) & ReadFlagString(g_adminFlags)) ? true : false;
}

public AdminFlagsChanged(Handle convar, const char[] oldValue, const char[] newValue)
{
	strcopy(g_adminFlags, sizeof(g_adminFlags), newValue);
}

public Prune()
{
	KvRewind(g_kv);
	if (!KvGotoFirstSubKey(g_kv))
	{
		return;
	}

	for(;;)
	{
		if(GetTime() > KvGetNum(g_kv, "timestamp") + GetConVarInt(g_CvarLockTime))
		{
			if (KvDeleteThis(g_kv) < 1)
			{
				break;
			}
		} else if (!KvGotoNextKey(g_kv)) {
			break;
		}	
	}
}

public IsValidAndAlive(int client)
{
	if (client > 0 && client <= MaxClients && IsClientInGame(client) && IsPlayerAlive(client))
		return true;
	else
		return false;
}

public IsValid(int client)
{
	if (client > 0 && client <= MaxClients && IsClientInGame(client))
		return true;
	else
		return false;
}

public protect(int client)
{
	int entity = getClientEntity(client);
	
	if (IsValidEntity(entity))
	{
		SetEntProp(entity, Prop_Data, "m_takedamage", 0, 1);
	}
}

public unprotect(int client)
{
	int entity = getClientEntity(client);
	
	if (IsValidEntity(entity))
	{
		SetEntProp(entity, Prop_Data, "m_takedamage", 2, 1);
	}
}

public getClientEntity(int client)
{
	int userid = GetClientUserId(client);
	return GetClientOfUserId(userid);
}